/**CFile***********************************************************************

  FileName    [simTime.c]

  PackageName [sim]

  Synopsis    [Timers handling]

  Description [Internal procedures included in this module:
		<ul>
		<li> <b>SimDllStartTimer()</b> starts a timer
		<li> <b>SimDllStopTimer()</b> stops a timer
		<li> <b>SimDllSetTimout()</b> sets the timeout
		</ul>]
		
  SeeAlso     [sim.h]

  Author      [Armando Tacchella]

  Copyright   [Copyright (c) 2000 by DIST - Universita' di Genova, Italia.  
               All Rights Reserved. This software is for educational
               purposes only.  Permission is given to academic
               institutions to use, copy, and modify this software and
               its documentation provided that this introductory
               message is not removed, that this software and its
               documentation is used for the institutions' internal
               research and educational purposes, and that no monies
               are exchanged. No guarantee is expressed or implied by
               the distribution of this code.  Permission to
               distribute this code is given to ITC-IRST,
               provided that the code is distributed as is.
               Send bug-reports and/or questions to: sim@mrg.dist.unige.it]

  Revision    [v. 1beta]

******************************************************************************/


#include "simInt.h"


/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

#ifdef HAVE_SYS_TIME_H
#ifdef HAVE_SYS_RESOURCE_H
#ifdef HAVE_UNISTD_H

/* A lot of stuff, needed by getrusage, get/setrlimit to work properly. */
#define HAVE_TIMER

#endif
#endif
#endif


/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis    [Starts a timer.]

  Description [Sets the specified timer to the current CPU resource usage.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
void
SimDllStartTimer(
  int             timer)
{
  
#ifdef HAVE_TIMER
  struct rusage t;

  getrusage(RUSAGE_SELF, &t);
  SimTimer[timer] = (float)t.ru_utime.tv_usec / 1.0e6;
  SimTimer[timer] += (float)t.ru_utime.tv_sec;
#endif

  return;

} /* End of SimDllStartTimer. */


/**Function********************************************************************

  Synopsis    [Stops a timer.]

  Description [Sets the specified timer to the current CPU resource usage
               minus the old timer value.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
void
SimDllStopTimer(
  int             timer)
{
  
#ifdef HAVE_TIMER
  struct rusage t;
  float         now;

  getrusage(RUSAGE_SELF, &t);
  now = ((float)t.ru_utime.tv_usec / 1.0e6) + (float)t.ru_utime.tv_sec;
  if (now <= SimTimer[timer]) { 
    SimTimer[timer] = 0.0;
  } else {
    SimTimer[timer] = now - SimTimer[timer];
  }
#endif

  return;

} /* End of SimDllStopTimer. */


/**Function********************************************************************

  Synopsis    [Sets the timeout on the process.]

  Description [Sets the specified timeout using the CPU resource limit
               mechanism.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
void
SimDllSetTimeout(
  int             tmout)
{
  
#ifdef HAVE_TIMER
  struct rlimit r;          
  
  if (tmout > 0) {
    getrlimit(RLIMIT_CPU, &r);
    r.rlim_cur = tmout;       
    setrlimit(RLIMIT_CPU, &r);
  }
#endif
  
  return;
  
} /* End of SimSetTimeout. */






