/*********************************************************/
/*  GSAT - Greedy Local Search Satisfiability Procedure  */
/*                                                       */
/*  Bart Selman and Henry Kautz                          */
/*                                                       */
/*                                                       */
/*  For information about this program, contact          */
/*      selman@research.att.com                          */
/*      kautz@research.att.com                           */
/*********************************************************/

#ifdef SYSVR4
#include <sys/times.h>
#include <limits.h>
#define NO_USLEEP
#endif

/**************************/
/*  globals.c -- GSAT     */
/**************************/

#define KEYBITS 64

/***************************/
/* ANSI/K&R compatibility  */
/***************************/

#ifndef PROTO

#ifndef NeedFunctionPrototypes
#if defined(FUNCPROTO) || __STDC__ || defined(__cplusplus) || defined(c_plusplus)
#define NeedFunctionPrototypes 1
#else
#define NeedFunctionPrototypes 0
#endif
#endif /* NeedFunctionPrototypes */

#  if NeedFunctionPrototypes
#    define PROTO( x )  x
#    define PARAMS( x )
#    define EXTERN_FUNCTION( fun, args ) extern fun args
#    define STATIC_FUNCTION( fun, args ) static fun args
#  else
#    define PROTO( x )
#    define PARAMS( x ) x 
#    define EXTERN_FUNCTION( fun, args ) extern fun ()
#    define STATIC_FUNCTION( fun, args ) static fun ()
#  endif

#endif /* PROTO */


/**************************/
/*  Standard Includes     */
/**************************/

#if NeedFunctionPrototypes || defined(sun)
#include <stdlib.h>
#endif

#include <malloc.h>
#include <stdio.h>
#ifndef SEEK_END
#define SEEK_END 2
#endif
#include <signal.h>
#include <string.h>
#include <math.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/utsname.h>
#include <stdlib.h>

/***************/
/*  Macros     */
/***************/

/* Math macros */
long scratch;		/* integer absolute value or var name */
#define abs_val(x) ((scratch=(x))<0 ? (-(scratch)) : (scratch))

#define BIG 0x7fffffff  /* 2**31 - 1 */

/* Parameter macros */
#define LINE_LENGTH 10		/* number of elements to print on each line when printing arrays */
#define MAXLINE 128		/* max number char for fgets function */
#define MAX_DIAGNOSTICS 10	/* max num output lines per diagnositic print function */  

#define PROGRAM_NAME "program: gsat version 41, July 1995\n"
#define PROGRAM_VERSION "41"

/****************************/
/*  Dynamic Data Structures */
/****************************/

typedef struct wff_str {
  long lit;			/* contains a lit OR the # of lits in following clause  */
  long next;			/* next clause containing the lit OR the weight of the clause */
				/*    lit:     #lits  lit1 lit2 lit3 ... */
				/*    next:    weight c1   c2   c3   ... */
				/* where c1 = index of start of next clause containing */
				/* lit1 or -lit1, etc. */
} *wff_str_ptr;

typedef struct var_str {
    long name;			/* same as the index of the variable */
    long value;	                /* new (possibly shorter name */
    long first;			/* index into wff of first clause containing var */
} *var_str_ptr;        

/****************************/
/*  Current State           */
/****************************/

#define FLAG_FORMAT_F 1
#define FLAG_FORMAT_KF 2
#define FLAG_FORMAT_CNF 3
#define FLAG_FORMAT_LISP 4
	  
/****************************/
/*  Dynamic Data Structures */
/****************************/

wff_str_ptr wff;		/* wff contains the clauses in the format */
				/*    #lits, lit1, ..., litn */
				/* First clause begins at index wff[1]. */
				/* size = nlits + nclauses + 1 */

var_str_ptr assign;		/* assign is global array containing the assignment */
				/* variable n stored at assign[n] */
				/* size = nvars + 1 */

long * clause_index_to_num;	/* array mapping clause indexes to clause numbers,
				   where indexes correspond to wff[] indexes */

long * clause_num_to_index = NULL;	/* array mapping clause numbers to clause indexes,
					   where indexes correspond to wff[] indexes */

/****************************/
/*  Current State           */
/****************************/

long nvars,			/* number of variables */
  nclauses,			/* number of clauses */
  nlits,                        /* number of literals  */
  newvars;			/* number of renamed variables */

/**************************/
/*  Parameters            */
/**************************/

char program_name[] = PROGRAM_NAME ;

FILE * try_stat_filep;

char wff_file[MAXLINE], convert_file[MAXLINE];

int
  flag_format;		        /* Indicates format of input file */

/**************************/
/*  Misc                  */
/**************************/

char ss[10000] = "Error message string not initialized";	/* scratch string */


int
interactive()
     /* Returns 1 iff stdin is a tty */
{
    return isatty(fileno(stdin));
}

void
exit_maybe()
     /* Error exit iff stdin is not a tty (i.e. not interactive) */
{
    if (!interactive()) exit (-1);
}

int
crash_and_burn(PROTO(char *) msg)
PARAMS( char * msg; )
{
    FILE *fp_report;
    char mailmsg[4000];

    printf("\n\nERROR - ERROR - ERROR - ERROR - ERROR - ERROR\n");
    printf(msg);
    printf("ERROR - ERROR - ERROR - ERROR - ERROR - ERROR\n");
    
    exit(-1);
}

void
crash_maybe(PROTO(char *) msg)
   /* Noisy error exit iff stdin is not a tty; otherwise, just print 
      error messge */
PARAMS( char * msg; )
{
    if (interactive()) {
	printf(msg);
    }
    else {
	crash_and_burn(msg);
    }
}

int
file_exists( PROTO(char *) fname)
     /* Returns 1 iff file named fname is readable */
PARAMS( char * fname; )
{
    FILE * fp;

    if ((fp = fopen(fname, "r")) == NULL) return 0;
    (void) fclose(fp);
    return 1;
}

void
get_input_file_name( PROTO( char * ) prompt, PROTO( char * ) name )
     /* Print prompt and read a line from stdin naming a file;
	check that the file exists; if not, then if interactive
	input, reprompt and reread; otherwise error exit. */
PARAMS( char *  prompt; char *  name; )
{
    while (1) {
	printf( prompt );
	if (gets( name ) == NULL) {
	    crash_maybe("Unexpected EOF on stdin!\n");
	    exit(-1);
	}
	if (file_exists( name )) return;
	crash_maybe("Error: cannot open file\n");
	exit(-1);
    }
}

void
get_symbol( PROTO( char * ) prompt, PROTO( char * ) dest)
     /* Print prompt and read a line; put the first (whitespace delimited)
	word in dest; if line is all whitespace, leave dest unchanged.
	If gets fails on end of file, error exit. */
PARAMS( char * prompt; char * dest; )
{
    char inputline[128];
    char dummy[128];

    printf(prompt);
    if (gets( inputline ) == NULL) {
	crash_maybe("Unexpected EOF on stdin!\n");
	exit(-1);
    }
    if (sscanf(inputline, " %s", dummy)==1)
      strcpy(dest, dummy);
}
    
int
empty_string( PROTO( char * ) str )
     /* Return 1 iff str is empty or only whitespace */
PARAMS( char * str; )
{
    char dummy[128];
    return (sscanf(str, " %s", dummy)!=1);
}

/********************/
/*  Initializing    */
/********************/

void
allocate_memory()
{
    wff = (wff_str_ptr) malloc ((size_t)((nlits + nclauses + 1) * (sizeof(struct wff_str))));
    assign = (var_str_ptr) malloc ((size_t)(nvars + 1) * (sizeof(struct var_str)));
}

void
init_pointers()
{
    long i, j, len_clause, alit, clause_index;
    var_str_ptr var_ptr;
    wff_str_ptr wff_ptr;

    /* initialize assign */
    for (i = 0, var_ptr=assign; i <= nvars; i++, var_ptr++){
	var_ptr->name = i;
	var_ptr->value = -1;
	var_ptr->first= -1;
    }

    /* Initialize the next field in wff, previously */
    /* set to all -1, and the first field in assign */

    wff_ptr = &wff[1];
    clause_index = 1;
    for (i=1; i<=nclauses; i++) {
	len_clause = wff_ptr->lit;
	wff_ptr++;

	for (j=1; j<=len_clause; j++) {
	    alit = abs_val(wff_ptr->lit);
	    wff_ptr->next = assign[alit].first;
	    assign[alit].first = clause_index;
	    wff_ptr ++;
	}
	clause_index += len_clause + 1;
    }
}

/********************/
/*  Printing        */
/********************/

long
length_of_clause_num(PROTO(long) clause_num)
PARAMS( long clause_num; )
{
    if (clause_num_to_index == NULL)
      crash_and_burn("ERROR: clause_num_to_index not initialized\n");
    if (clause_num < 1 || clause_num > nclauses)
      crash_and_burn("ERROR: clause_num out of range\n");
    return (wff[ clause_num_to_index[clause_num] ].lit );
}

long
lit_of_clause_num( PROTO(long) lit_num, PROTO(long) clause_num)
PARAMS(long  lit_num; long clause_num; )
{
    if (clause_num_to_index == NULL)
      crash_and_burn("ERROR: clause_num_to_index not initialized\n");
    if (clause_num < 1 || clause_num > nclauses)
      crash_and_burn("ERROR: clause_num out of range\n");
    if (lit_num < 1 || lit_num > length_of_clause_num(clause_num))
      crash_and_burn("ERROR: lit_num out of range\n");
    return (wff[ clause_num_to_index[clause_num] + lit_num ].lit );
}

void
print_wff()  /* print the raw wff array */
{
  long i;
  printf("raw wff:\nindex  lit   next\n");
  for (i=1; i<=(nlits + nclauses) && i<=MAX_DIAGNOSTICS; i++){
    printf("  %ld      %ld       %ld\n", i, wff[i].lit, wff[i].next);
  }
}


void
print_clause(PROTO(long) clause_index)
PARAMS(long clause_index;)
{
  long clause_len, i;
  long lit;

  clause_len = wff[clause_index].lit;
  for (i=1; i<=clause_len; i++){
      lit = wff[clause_index + i].lit;
      printf(" %ld ", lit );
  }
  printf("\n");
}

void
print_wff_clauses()  /* print the wff in clausal form */
{
  long clause_index, i;

  printf("wff clauses:\n");
  
  clause_index=1;
  for (i = 1; i <= nclauses && i<=MAX_DIAGNOSTICS; i++){
    printf("clause %ld is ", i);
    print_clause(clause_index);
    clause_index += 1 + wff[clause_index].lit;
  }
}

     
/*************************/
/*  Output in Wff Format */
/*************************/

long numero;
#define rename(x) ((numero=(x))>0 ? (assign[numero].value) : (-(assign[-(numero)].value)))

void
output_f_format(PROTO(FILE *) fp)
PARAMS(FILE *fp;)
{
    long i, j, index, clause_len;

    printf("Writing f-format\n");

    index = 1;
    for (i=1; i<=nclauses; i++) {
	clause_len = wff[index].lit;
	for (j=1; j<= clause_len; j++)
	    fprintf(fp, " %ld",rename(wff[index+j].lit));
	index += clause_len + 1;
	fprintf(fp, "\n");	
    }
    fprintf(fp, "%%\n0\n");
}

void
output_f_lisp_format(PROTO(FILE *) fp)
PARAMS(FILE *fp;)
{
    long i, j, index, clause_len;

    printf("Writing f-lisp-format\n");

    index = 1;
    for (i=1; i<=nclauses; i++) {
	clause_len = wff[index].lit;
	fprintf(fp, "(");
	for (j=1; j<= clause_len; j++)
	    fprintf(fp, " %ld", rename(wff[index+j].lit));
	index += clause_len + 1;
	fprintf(fp, ")\n");	
    }
}

void
output_kf_format(PROTO(FILE *) fp)
PARAMS( FILE *fp; )
{
    long i, j, index, clause_len;

    printf("Writing f-format\n");

    fprintf(fp, "%ld %ld\n", newvars, nclauses);
    index = 1;
    for (i=1; i<=nclauses; i++) {
	clause_len = wff[index].lit;
	fprintf(fp, "%ld ", clause_len);
	for (j=1; j<= clause_len; j++)
	    fprintf(fp, " %ld ",rename(wff[index+j].lit));
	fprintf(fp, "\n");
	index += clause_len + 1;
    }
}

void
output_cnf_format(PROTO(FILE *) fp)
PARAMS( FILE *fp; )
{
    long i, j, index, clause_len;

    printf("Writing cnf-format\n");

    fprintf(fp, "p cnf %ld %ld\n", newvars, nclauses);
    index = 1;
    for (i=1; i<=nclauses; i++) {
	clause_len = wff[index].lit;
	for (j=1; j<= clause_len; j++){
	    fprintf(fp, "%ld ",rename(wff[index+j].lit));
	}
	fprintf(fp, "0\n");
	index += clause_len + 1;
    }
}

void
output_converted_wff()
{
    FILE *fp;
    if ((fp = fopen(convert_file, "w"))==NULL) {
	sprintf(ss, "ERROR: cannot open output file %s\n", convert_file);
	crash_and_burn(ss);
    }

    if (flag_format == FLAG_FORMAT_KF)
      output_kf_format(fp);
    else if (flag_format == FLAG_FORMAT_F)
      output_f_format(fp);
    else if (flag_format == FLAG_FORMAT_CNF)
      output_cnf_format(fp);
    else if (flag_format == FLAG_FORMAT_LISP)
      output_f_lisp_format(fp);
    else if (0==strcmp(".f", &(convert_file[strlen(convert_file)-2])))
      output_f_format(fp);
    else if (0==strcmp(".np", &(convert_file[strlen(convert_file)-3])))
      output_f_format(fp);
    else if (0==strcmp(".kf", &(convert_file[strlen(convert_file)-3])))
      output_kf_format(fp);
    else if (0==strcmp(".cnf", &(convert_file[strlen(convert_file)-4])))
      output_cnf_format(fp);
    else if (0==strcmp(".lisp", &(convert_file[strlen(convert_file)-5])))
      output_f_lisp_format(fp);
    else {
	sprintf(ss, "ERROR: unknown output format\n");
	crash_and_burn(ss);
    }

    fclose(fp);
}


/**************************/
/*  Parsing Parameters    */
/**************************/

void
parse_parameters()
{
    char inputline[MAXLINE];
    char word1[MAXLINE];

    printf(program_name);

    get_input_file_name("wff_file: ", wff_file);
    printf("wff_file: %s\n", wff_file);

    sprintf(convert_file,"new clauseset.f");
    printf("convert file (default = %s): ");
    get_symbol("",convert_file);
    printf("convert_file: %s\n", convert_file);
  
    /* Get and print optional parameters */
    while (1) {
	printf("option (? for help): ");
	if (gets(inputline)==NULL) break;
	if (sscanf(inputline, " %s", word1)!=1) break;
	if (strcmp(word1,"?")==0) {
	    printf("  cnf = output wff in cnf-format\n");
	    printf("  f = output wff in f-format\n");
	    printf("  kf = output wff in kf-format\n");
	    printf("  lisp = output wff in f-list-format\n");
	    printf("  np = output wff in f-format\n");
	}
	else if (strcmp(word1,"np")==0 || strcmp(word1,"f")==0) {
	    printf(" option: f-format \n");
	    flag_format = FLAG_FORMAT_F; }
	else if (strcmp(word1,"kf")==0) {
	    printf(" option: kf-format \n");
	    flag_format = FLAG_FORMAT_KF; }
	else if (strcmp(word1,"cnf")==0) {
	    printf(" option: cnf-format \n");
	    flag_format = FLAG_FORMAT_CNF; }
	else if (strcmp(word1,"lisp")==0) {
	    printf(" option: lisp-format \n");
	    flag_format = FLAG_FORMAT_LISP; }
	else {
	    sprintf(ss, " ERROR: unknown option %s\n", inputline);
	    crash_maybe(ss);
	}
    }
}

/**************************/
/*  Parsing Input  wff    */
/**************************/

long
white(PROTO(long) c)
PARAMS( long c; )
{
    return (c==' ' || c=='\n' || c=='\t' || c=='\f');
}

long
space_or_paren(PROTO(long) c)
PARAMS( long c; )
{
    return (c==' ' || c=='(' || c=='\t' || c==')');
}



long
read_in_f_format()	
{
    FILE *fp;
    long c;
    char field[MAXLINE];
    long index, wffindex, clause_start, clause_length;
    long lit, var;
    long i, tautologous_clause, repeated_literal;
    long clause_number;
    
    nvars = 0;
    nclauses = 0;
    nlits = 0;

    printf("Trying to read f-format input\n");

    /* Scan once to calculate nvars, nlits, nclauses */
    if ((fp = fopen(wff_file, "r"))==NULL){
	sprintf(ss, "ERROR: cannot open wff file %s\n", wff_file);
	crash_and_burn(ss);
    }

    c = ' ';
    while (space_or_paren(c)) c=getc(fp);
    while (c != EOF && c != '%'){
	nclauses ++;
	if (c==EOF){ fclose(fp); return 0; }
	while (c != '\n'){
	    index=0;

	    if (!(c=='-' || (c >= '0' && c <= '9'))) {
		fclose(fp); return 0; }

	    while (c=='-' || (c >= '0' && c <= '9')){
		field[index++]=c;
		c=fgetc(fp);
	    }
	    field[index] = 0;
	    lit = atoi(field);
	    var = abs_val(lit);
	    if (var > nvars) nvars = var;
	    nlits ++;
	    while (space_or_paren(c)) c=getc(fp);
	    if (c==EOF){
		fclose(fp); return 0;
	    }
	}
	c=getc(fp);
	while (space_or_paren(c)) c=getc(fp);
    }
    fclose(fp);

    printf("nvars = %ld\nnclauses = %ld\nnlits = %ld\n", nvars, nclauses, nlits);    
    allocate_memory();

    /* Second scan, to fill in the wff array */
    fp = fopen(wff_file, "r");
    c = ' ';
    wffindex = 1;
    while (white(c)) c=getc(fp);
    clause_number = 0;
    while (c != EOF && c != '%'){
	clause_number++;
	clause_start = wffindex++;
	clause_length = 0;
	if (c==EOF){
	    return  0;
	}
	tautologous_clause = 0;
	while (c != '\n'){
	    index=0;
	    while(c=='-' || (c >= '0' && c <= '9')){
		field[index++]=c;
		c=fgetc(fp);
	    }
	    field[index] = 0;
	    lit = atoi(field);

	    /* DEBUG(printf("read %s=%ld ", field, lit)); */

	    repeated_literal = 0;
	    if (! tautologous_clause){
		for (i=1; i<= clause_length; i++){
		    if (lit == wff[wffindex - i].lit) {
			repeated_literal = 1;
			printf("Warning!  Repeated literal %ld in clause %ld\n",
			       lit, clause_number);
		    }
		    if (lit == -wff[wffindex - i].lit) {
			tautologous_clause = 1;
			wffindex = clause_start;
			nclauses--;
			printf("Warning!  Complementary literals %ld and %ld in clause %ld\n",
			       lit, -lit, clause_number);
		    }
		}
	    }

	    if (! tautologous_clause && ! repeated_literal){
		wff[wffindex].lit = lit;
		wff[wffindex].next = -1;
		wffindex++;
		clause_length++;
	    }

	    while (space_or_paren(c)) c=getc(fp);
	    if (c==EOF){
		fclose(fp); return 0;
	    }
	}
	if (! tautologous_clause){
	    wff[clause_start].lit = clause_length;
	    wff[clause_start].next = -1;
	}

	if (clause_length == 0){
	    printf("Warning!  Wff contains empty (FALSE) clauses, and must be unsat!\n");
	}

	/* DEBUG(print_clause(clause_start)); */

	c=getc(fp);
	while (space_or_paren(c)) c=getc(fp);
    }
    fclose(fp);
    return 1;
}


long
read_in_f_lisp_format()	
{
    FILE *fp;
    long c;
    char field[MAXLINE];
    long index, wffindex, clause_start, clause_length;
    long lit, var;
    long i, tautologous_clause, repeated_literal;
    long clause_number;
    
    nvars = 0;
    nclauses = 0;
    nlits = 0;

    printf("Trying to read f-lisp-format\n");

    /* Scan once to calculate nvars, nlits, nclauses */
    if ((fp = fopen(wff_file, "r"))==NULL){
	sprintf(ss, "ERROR: cannot open wff file %s\n", wff_file);
	crash_and_burn(ss);
    }
    c = ' ';
    while (white(c)) c=getc(fp);
    while (c != EOF && c != '%'){
	if (c != '('){
	    fclose(fp); return 0;
	}
	nclauses ++;
	c=getc(fp);
	while (white(c)) c=getc(fp);
	if (c==EOF){
	    fclose(fp); return 0;
	}
	while (c != ')'){
	    index=0;

	    if (!(c=='-' || (c >= '0' && c <= '9'))){
		fclose(fp); return 0;
	    }
	    while (c=='-' || (c >= '0' && c <= '9')){
		field[index++]=c;
		c=fgetc(fp);
	    }
	    field[index] = 0;
	    lit = atoi(field);
	    var = abs_val(lit);
	    if (var > nvars) nvars = var;
	    nlits ++;
	    while (white(c)) c=getc(fp);
	    if (c==EOF){
		fclose(fp); return 0;
	    }
	}
	c=getc(fp);
	while (white(c)) c=getc(fp);
    }
    fclose(fp);

    printf("nvars = %ld\nnclauses = %ld\nnlits = %ld\n", nvars, nclauses, nlits);    
    allocate_memory();

    /* Second scan, to fill in the wff array */
    fp = fopen(wff_file, "r");
    c = ' ';
    wffindex = 1;
    while (white(c)) c=getc(fp);
    clause_number = 0;
    while (c != EOF && c != '%'){
	if (c != '('){
	    fclose(fp); return 0;
	}
	clause_number++;
	clause_start = wffindex++;
	clause_length = 0;
	c=getc(fp);
	while (white(c)) c=getc(fp);
	if (c==EOF){
	    fclose(fp); return 0;
	}
	tautologous_clause = 0;
	while (c != ')'){
	    index=0;
	    while(c=='-' || (c >= '0' && c <= '9')){
		field[index++]=c;
		c=fgetc(fp);
	    }
	    field[index] = 0;
	    lit = atoi(field);

	    /* DEBUG(printf("read %s=%ld ", field, lit)); */

	    repeated_literal = 0;
	    if (! tautologous_clause){
		for (i=1; i<= clause_length; i++){
		    if (lit == wff[wffindex - i].lit) {
			repeated_literal = 1;
			printf("Warning!  Repeated literal %ld in clause %ld\n",
			       lit, clause_number);
		    }
		    if (lit == -wff[wffindex - i].lit) {
			tautologous_clause = 1;
			wffindex = clause_start;
			nclauses--;
			printf("Warning!  Complementary literals %ld and %ld in clause %ld\n",
			       lit, -lit, clause_number);
		    }
		}
	    }

	    if (! tautologous_clause && ! repeated_literal){
		wff[wffindex].lit = lit;
		wff[wffindex].next = -1;
		wffindex++;
		clause_length++;
	    }
	    
	    while (white(c)) c=getc(fp);
	    if (c==EOF){
		fclose(fp); return 0;
	    }
	}
	if (! tautologous_clause){
	    wff[clause_start].lit = clause_length;
	    wff[clause_start].next = -1;
	}
	
	/* DEBUG(print_clause(clause_start)); */

	c=getc(fp);
	while (white(c)) c=getc(fp);
    }
    fclose(fp);
    return 1;
}

long
read_in_kf_format()
{
    long i, j, len_clause, lit;
    FILE * fp;
    long tautologous_clause, repeated_literal;
    long clause_number, number_input_clauses;
    long wffstart, wffindex;

    printf("Trying to read kf-format\n");

    if ((fp = fopen(wff_file, "r"))==NULL){
	sprintf(ss, "ERROR: cannot open wff file %s\n", wff_file);
	crash_and_burn(ss);
    }

    if (fscanf(fp,"%ld %ld", &nvars, &nclauses ) != 2) {
	fclose(fp);
	return 0;
    }

    if (nvars <= 0){
	fclose(fp);
	return 0;
    }

    number_input_clauses = nclauses;

    /* determine number of lits in wff */
    nlits = 0;
    for (i = 1; i <= nclauses ; i++) {
	if (fscanf(fp, "%ld", &len_clause)!=1){
	    printf("WARNING!  Unexpected character in file; maybe this is np format?\n");
	    fclose(fp);
	    return 0;
	}
	if (len_clause < 0) {
	    fclose(fp);
	    return 0;
	}
	nlits = nlits + len_clause;
	for (j = 1; j <= len_clause ; j++){
	    if (fscanf(fp, "%ld", &lit)!=1){
		fclose(fp);
		return 0;
	    }
	}
    }
    if (fscanf(fp, "%ld", &len_clause)==1) {
	fclose(fp);
	return 0;
    }
    fclose(fp);
    printf("nvars = %ld\nnclauses = %ld\nnlits = %ld\n", nvars, nclauses, nlits);    
    allocate_memory();

    /* read in wff into .lit field, setting .next field to -1 */
    fp = fopen(wff_file, "r");
    fscanf(fp,"%ld %ld", &i, &i);
    wffindex = 1;
    for (clause_number=1; clause_number<=number_input_clauses; clause_number++){
	tautologous_clause = 0;
	fscanf(fp,"%ld", &len_clause);
	wffstart = wffindex;
	wff[wffstart].lit = len_clause;
	wff[wffstart].next = -1;
	wffindex++;
	
	for (j=1; j<=len_clause; j++){
	    repeated_literal = 0;	
	    fscanf(fp, "%ld", &lit);
	    if (! tautologous_clause){
		for (i=wffstart+1; i<wffindex; i++){
		    if (lit == wff[i].lit) {
			repeated_literal = 1;
			wff[wffstart].lit --;
			printf("Warning!  Repeated literal %ld in clause %ld\n",
			       lit, clause_number);
		    }
		    if (lit == -wff[i].lit) {
			tautologous_clause = 1;
			nclauses--;
			wffindex = wffstart;
			printf("Warning!  Complementary literals %ld and %ld in clause %ld\n",
			       lit, -lit, clause_number);
		    }
		}
	    }
	    if (! tautologous_clause && ! repeated_literal){
		wff[wffindex].lit = lit;
		wff[wffindex].next = -1;
		wffindex++;
	    }
	}
    }

    fclose(fp);
    return 1;
}

long
read_in_cnf_format()
{
    long i, j, len_clause, lit;
    FILE * fp;
    long tautologous_clause, repeated_literal;
    long clause_number, number_input_clauses;
    long wffstart, wffindex;
    long lastc;
    long nextc;

    printf("Trying to read cnf-format\n");

    if ((fp = fopen(wff_file, "r"))==NULL){
	sprintf(ss, "ERROR: cannot open wff file %s\n", wff_file);
	crash_and_burn(ss);
    }
    
    while ((lastc = fgetc(fp)) == 'c')
      {
	  while ((nextc = fgetc(fp)) != EOF && nextc != '\n');
      }
    ungetc(lastc,fp);

    if (fscanf(fp, "p cnf %i %i",&nvars,&nclauses) != 2){
	fclose(fp);
	return 0;
    }

    if (nvars <= 0){
	fclose(fp);
	return 0;
    }

    number_input_clauses = nclauses;
    /* determine number of lits in wff */
    nlits = 0;
    for (i = 1; i <= nclauses ; i++) {
	if (fscanf(fp, "%ld", &lit)!=1) { fclose(fp); return 0; }
	while (lit){
	    nlits++;
	    if (fscanf(fp, "%ld", &lit)!=1) { fclose(fp); return 0; }
	}
    }
    if (fscanf(fp, "%ld", &len_clause)==1) {
	fclose(fp);
	return 0;
    }
    fclose(fp);
    printf("nvars = %ld\nnclauses = %ld\nnlits = %ld\n", nvars, nclauses, nlits);    
    allocate_memory();

    /* read in wff into .lit field, setting .next field to -1 */
    fp = fopen(wff_file, "r");
    while ((lastc = fgetc(fp)) == 'c')
      {
	  while ((nextc = fgetc(fp)) != EOF && nextc != '\n');
      }
    ungetc(lastc,fp);

    if (fscanf(fp, "p cnf %i %i",&nvars,&nclauses) != 2){
	fclose(fp);
	return 0;
    }

    wffindex = 1;
    
    for (clause_number=1; clause_number<=number_input_clauses; clause_number++){
	tautologous_clause = 0;
	wffstart = wffindex;
	len_clause = 0;
	wff[wffstart].next = -1;
	wffindex++;

	fscanf(fp, "%ld", &lit);
	while (lit) {
	    repeated_literal = 0;	
	    if (! tautologous_clause){
		for (i=wffstart+1; i<wffindex; i++){
		    if (lit == wff[i].lit) {
			repeated_literal = 1;
			printf("Warning!  Repeated literal %ld in clause %ld\n",
			       lit, clause_number);
		    }
		    if (lit == -wff[i].lit) {
			tautologous_clause = 1;
			nclauses--;
			wffindex = wffstart;
			printf("Warning!  Complementary literals %ld and %ld in clause %ld\n",
			       lit, -lit, clause_number);
		    }
		}
	    }
	    if (! tautologous_clause && ! repeated_literal){
		wff[wffindex].lit = lit;
		wff[wffindex].next = -1;
		wffindex++;
		len_clause++;
	    }
	    fscanf(fp, "%ld", &lit);
	}
	wff[wffstart].lit = len_clause;
    }

    fclose(fp);
    return 1;
}


void
read_in()
{

int flag_input;

    if (0==strcmp(".f", &(wff_file[strlen(wff_file)-2])))
	  flag_input = FLAG_FORMAT_F;
    else if (0==strcmp(".np", &(wff_file[strlen(wff_file)-3])))
	  flag_input = FLAG_FORMAT_F;
    else if (0==strcmp(".lisp", &(wff_file[strlen(wff_file)-5])))
	  flag_input = FLAG_FORMAT_LISP;
    else if (0==strcmp(".kf", &(wff_file[strlen(wff_file)-3])))
	  flag_input = FLAG_FORMAT_KF;
    else if (0==strcmp(".cnf", &(wff_file[strlen(wff_file)-4])))
	  flag_input = FLAG_FORMAT_CNF;
    else flag_input=0;
   
    switch (flag_input){
      case FLAG_FORMAT_KF:
	read_in_kf_format() ||  crash_and_burn("Bad input file!\n");
	break;
      case FLAG_FORMAT_LISP:
	read_in_f_lisp_format() ||  crash_and_burn("Bad input file!\n");
	break;
      case FLAG_FORMAT_CNF:
	read_in_cnf_format() ||  crash_and_burn("Bad input file!\n");
	break;
      case FLAG_FORMAT_F:
	read_in_f_lisp_format() ||  
	  read_in_f_format() ||  
	    crash_and_burn("Bad input file!\n");
	break;
      default:
	read_in_cnf_format() ||  
	  read_in_f_lisp_format() ||  
	    read_in_kf_format() ||  
	      read_in_f_format() ||  
		crash_and_burn("Bad input file!\n");
	break;
    }
}

/*************************************/
/*  Creating the Initial Assignment  */
/*************************************/

void
init_assign_values() 
{
    long i,var,key_var;
    var_str_ptr iptr;

    iptr = &assign[1];

    key_var=0;
    for (i = 1; i <= KEYBITS; i++){
        iptr->value=0;
        if (iptr->first != -1){
	    key_var++;
	    iptr->value=i;
	}
       iptr++;
    }

    var=KEYBITS;

    for (i = KEYBITS+1; i <= nvars; i++){
        iptr->value=0;
        if (iptr->first != -1){
	    var++;
	    iptr->value = var;
	}
       iptr++;
    }
    newvars=var;
    printf("Present only keybits = %ld  vars = %ld\n",key_var,newvars-KEYBITS);    printf("Effective Constraintdness = %.2f\n",(float)nclauses/(key_var+newvars-KEYBITS));    
}

/***************/
/*  MAIN       */
/***************/

int
main(PROTO(long) argc, PROTO(char **) argv)
PARAMS( char ** argv; long argc; )
{
  setbuf(stdout,NULL);

  flag_format = 0;

  parse_parameters();

  printf("Reading Wff\n");

  read_in();

  printf("Initializing Pointers\n");
  
  init_pointers();

  printf("Remapping Variables\n");

  init_assign_values();

  printf("Writing Wff\n");

  output_converted_wff();

  return(0);
}


